/*************************************************************************
 * The contents of this file are subject to the MYRICOM MX AND GM-2      *
 * MAPPING SOFTWARE AND DOCUMENTATION LICENSE (the "License"); User may  *
 * not use this file except in compliance with the License.  The full    *
 * text of the License can found in mapper directory in LICENSE.TXT      *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

#include <stdarg.h>

#include "lx.h"
#include "lx_map_file.h"
#include "lx_routing.h"
#include "lx_route_file.h"

lx_t lx;

static void usage (char*s)
{
  fprintf (stderr, "usage: %s <map file> [<route file>]\n", s);
  exit (0);
}

static int count_hops (lx_node_t*n, lx_node_t*to, int iport, int oport, lx_route_t*r)
{ 
  int i, p, last;

  insist (n && to && r);
  insist (iport >= 0 && iport < LX_HOST_SIZE);
  insist (oport >= 0 && oport < LX_HOST_SIZE);

  insist (n);
 
  last = lx_get_port (n, iport);
  n = lx_get_node (n, iport);
  insist (n);

  for (i = 0; i < r->length; i++)
  {
    insist (last + r->hops [i] >= 0 && last + r->hops [i] < LX_XBAR_SIZE);
    n->node_data.counts [last + r->hops [i]]++;
      
    p = lx_get_port (n, last + r->hops [i]);
    n = lx_get_node (n, last + r->hops [i]);
    insist (n);
    last = p;
  }
  
  insist (!r->length || (n == to && last == oport));
  return 1;
  except: return 0;
}

int main (int argc, char*argv [])
{
  lx_map_t*m;
  FILE*fp;
  int i, j, o, k, from, num_hosts, num_ports, line_count;
  int max_depth, num_routes, num_spines;
  lx_route_table_entry_t*entries, *e;
  lx_node_t*first, *n;  

  mi_verbose_flag = 0;

  if (argc < 2)
    usage (argv [0]);
  
  if (!(fp = fopen (argv [1], "r")))
  {
    fprintf (stderr, "couldn't open %s\n", argv [1]);
    usage (argv [0]);
  }
  
  lx_init (&lx);
  
  if (!lx_map_file_read (fp, &lx.maps [0], &first))
  {
    fprintf (stderr, "lx_map_file_read () failed\n");
    return 0;
  }

  m = lx.maps;
  fprintf (stderr, "%d hosts and %d xbars\n", m->num_hosts, m->num_xbars);

  if (!(fp = argc == 2 ? stdin : fopen (argv [2], "r")))
  {
    fprintf (stderr, "couldn't open %s\n", argv [2]);
    usage (argv [0]);
  }
  
  line_count = 0;
  
  while ((entries = lx_route_file_read_routes (fp, &from, &num_hosts, &num_ports, &num_routes, &line_count)))
  {
    insist (from >= 0 && from < m->num_hosts);
    insist (num_hosts == m->num_hosts);
    insist (num_ports > 0 && num_ports <= LX_HOST_SIZE);
    insist (num_routes > 0);
    
    for (i = 0; i < num_hosts; i++)
    {
      for (j = 0; j < num_ports; j++)
      {
	for (o = 0; o < num_ports; o++)
	{
	  for (k = 0; k < num_routes; k++)
	  {
	    e = lx_route_file_entry (entries, num_ports, num_hosts, num_routes, j, i, o, k);
	    insist (e);
	    if (e->valid && !count_hops (m->host_array [from], m->host_array [i], j, o, &e->route))
	      return 0;
	  }
	}
      }
    }
    free (entries);
  }

  if (!(max_depth = lx_routing_number_xbars_clos (m, &num_spines)))
  {    
    fprintf (stderr, "couldn't number spines\n");
    return 0;
  }
  fprintf (stderr, "max_depth is %d, %d xbars at spine\n", max_depth, num_spines);
  

  for (j = 0; j < max_depth; j++)
  {    
    printf ("level %d\n", j);
    for (i = 0; i < m->num_xbars; i++)
    {
      n = m->xbar_array [i];
      insist (n);
      if (lx_xbar_c (n)->level == j)
      {
	printf (lx_node_format " ", lx_node_args (n));
    
	for (k = 0; k < LX_XBAR_SIZE; k++)
	  printf ("%d ", n->node_data.counts [k]);
	printf ("\n");
      }      
    }
  }
  
  return 1;
  except: return 0;
}
